<?php
$citiesObj = new Cities();
// declare(strict_types=1);

class Cities 
{
    private PDO $db;
    private array $config;
    private array $cache = [];
    private const CACHE_DURATION = 3600; // 1 hour
    private const ALLOWED_SORT_COLUMNS = ['c.id', 'c.status', 'c.date_added', 'u.full_name', 'cl.name'];
    private const ALLOWED_SORT_TYPES = ['ASC', 'DESC'];

    public function __construct()
    {
        global $Config;
        $this->db = (new iplus())->connect();
        $this->config = $Config;
    }

    private function validateSort(string $sort, string $type): array
    {
        $sort = in_array($sort, self::ALLOWED_SORT_COLUMNS) ? $sort : 'c.date_added';
        $type = in_array(strtoupper($type), self::ALLOWED_SORT_TYPES) ? strtoupper($type) : 'ASC';
        return [$sort, $type];
    }

    private function getCacheKey(string $prefix, array $params): string
    {
        return md5($prefix . serialize($params));
    }

    public function getAllCities(?string $name = null, int $start = 0, int $itemsPerPage = 10, string $sort = 'c.date_added', string $type = 'ASC'): array
    {
        try {
            $cacheKey = $this->getCacheKey('cities_list', [$name, $start, $itemsPerPage, $sort, $type]);
            
            if (isset($this->cache[$cacheKey]) && $this->cache[$cacheKey]['expires'] > time()) {
                return $this->cache[$cacheKey]['data'];
            }

            [$sort, $type] = $this->validateSort($sort, $type);
            
            $sql = "
                SELECT 
                    c.id, c.status, c.date_added, c.added_by,
                    u.full_name, cl.name,
                    COUNT(*) OVER() as total_count
                FROM cities_management c
                LEFT JOIN users u ON c.added_by = u.id
                LEFT JOIN cities_management_langs cl ON c.id = cl.city_id AND cl.lang_code = :lang_code
                WHERE c.is_deleted = '0'
            ";
            
            $params = [];
            if ($name) {
                $sql .= " AND cl.name LIKE :name";
                $params[':name'] = "%{$name}%";
            }
            
            $sql .= " ORDER BY {$sort} {$type} LIMIT :start, :limit";
            $params[':start'] = $start;
            $params[':limit'] = $itemsPerPage;

            $stmt = $this->db->prepare($sql);
            foreach ($params as $key => &$val) {
                $stmt->bindValue($key, $val, is_int($val) ? PDO::PARAM_INT : PDO::PARAM_STR);
            }
            $lang_code = LANGUAGE;
        $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);
            $stmt->execute();
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $total = $data[0]['total_count'] ?? 0;

            $result = [
                'status' => 200,
                'data' => $data,
                'count' => (int)$total,
                'success' => true
            ];

            $this->cache[$cacheKey] = [
                'data' => $result,
                'expires' => time() + self::CACHE_DURATION
            ];

            return $result;
        } catch (PDOException $e) {
            error_log("Database error in getAllCities: " . $e->getMessage());
            return [
                'status' => 500,
                'error' => 'Database operation failed',
                'success' => false
            ];
        }
    }

    public function addEditCity(array $request): array
    {   
        try {
            $this->db->beginTransaction();
            
            $dateTime = date('Y-m-d H:i:s');
            $langs = json_decode($request['langs'] ?? '[]', true);
            
            if (empty($request['id'])) {
                $cityId = $this->insertCity($request, $dateTime);
            } else {
                $cityId = $this->updateCity($request);
            }
            
            if (!empty($langs)) {
                $this->updateCityLanguages($cityId, $langs);
            }
            
            $this->db->commit();
            return [
                'status' => 200,
                'data' => empty($request['id']) ? 'Added successfully' : 'Updated successfully',
                'success' => true
            ];
        } catch (PDOException $e) {
            $this->db->rollBack();
            error_log("Error in addEditCity: " . $e->getMessage());
            return [
                'status' => 500,
                'error' => 'Operation failed',
                'success' => false
            ];
        }
    }

    private function insertCity(array $request, string $dateTime): int
    {
        $stmt = $this->db->prepare("
            INSERT INTO cities_management (status, added_by, date_added) 
            VALUES (:status, :added_by, :date_added)
        ");
        
        $stmt->execute([
            ':status' => $request['status'],
            ':added_by' => $request['user_id'],
            ':date_added' => $dateTime
        ]);
        
        return (int)$this->db->lastInsertId();
    }

    private function updateCity(array $request): int
    {
        if (isset($request['status'])) {
            $stmt = $this->db->prepare("
                UPDATE cities_management 
                SET status = :status 
                WHERE id = :id
            ");
            $stmt->execute([
                ':status' => $request['status'],
                ':id' => $request['id']
            ]);
        }
        return (int)$request['id'];
    }

    private function updateCityLanguages(int $cityId, array $langs): void
    {
        $stmt = $this->db->prepare("
            INSERT INTO cities_management_langs (lang_code, name, city_id)
            VALUES (:lang_code, :name, :city_id)
            ON DUPLICATE KEY UPDATE name = VALUES(name)
        ");
        
        foreach ($langs as $lang) {
            $stmt->execute([
                ':lang_code' => $lang['lang_code'],
                ':name' => $lang['name'],
                ':city_id' => $cityId
            ]);
        }
    }

    public function deleteCity(array $ids): array
    {
        if (empty($ids) || !is_array($ids)) {
            return [
                'status' => 400,
                'success' => false,
                'data' => "Invalid input"
            ];
        }

        try {
            $placeholders = str_repeat('?,', count($ids) - 1) . '?';
            $stmt = $this->db->prepare("
                UPDATE cities_management 
                SET is_deleted = '1'
                WHERE id IN ($placeholders)
            ");
            
            $stmt->execute($ids);
            $deletedCount = $stmt->rowCount();

            return [
                'status' => $deletedCount ? 200 : 404,
                'success' => $deletedCount > 0,
                'data' => $deletedCount ? "{$deletedCount} items deleted successfully" : "No items found"
            ];
        } catch (PDOException $e) {
            error_log("Error in deleteCity: " . $e->getMessage());
            return [
                'status' => 500,
                'success' => false,
                'data' => "Operation failed"
            ];
        }
    }

    public function getCityInfo(int $id): array
    {  
        try {
        
            $stmt = $this->db->prepare("
                SELECT 
                    c.id, c.date_added, c.status,
                    GROUP_CONCAT(
                        JSON_OBJECT(
                            'lang_code', cl.lang_code,
                            'name', cl.name
                        )
                    ) as langs
                FROM cities_management c
                LEFT JOIN cities_management_langs cl ON c.id = cl.city_id
                WHERE c.is_deleted = '0' AND c.id = :id
                GROUP BY c.id
            ");
            
            $stmt->execute([':id' => $id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            // $result = json_decode($result['langs'], true);
            if (!$result) {
                return [
                    'status' => 404,
                    'success' => false,
                    'data' => 'City not found'
                ];
            }

            // Convert the GROUP_CONCAT result to an array
            $result['langs'] = json_decode('['.$result['langs'].']', true);
            return [
                'status' => 200,
                'success' => true,
                'data' => $result
            ];
        } catch (PDOException $e) { 
            error_log("Error in getCityInfo: " . $e->getMessage());
            return [
                'status' => 500,
                'success' => false,
                'data' => 'Operation failed'
            ];
        }
    }
}
